<?php
/*--------------------------------------------------------------
   GambioVisitorsOnlineCount.php 2021-07-19
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;

use Doctrine\DBAL\Connection;
use Gambio\Admin\Modules\Statistics\App\Data\Factory as DataFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory as OverviewFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Options;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data;
use NumberFormatter;

class GambioVisitorsOnlineCount extends GambioWidgetDefinition
{
    private const TYPE = 'gambio.visitors.online.count';
    
    private const VERSION = '0.1.0';
    
    private const ICON = "data:image/png;base64,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";
    
    private const WIDGET_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Besucher online',
        self::LANGUAGE_CODE_ENGLISH => 'Visitors online'
    ];
    
    /**
     * @var DataFactory
     */
    private $dataFactory;
    
    /**
     * @var Connection
     */
    private $connection;
    
    /**
     * @var NumberFormatter
     */
    private $numberFormatter;
    
    
    /**
     * @inheritDoc
     */
    public function __construct(
        OverviewFactory $overviewFactory,
        DataFactory $dataFactory,
        Connection $connection,
        NumberFormatter $numberFormatter
    ) {
        $this->dataFactory     = $dataFactory;
        $this->connection      = $connection;
        $this->numberFormatter = $numberFormatter;
        
        parent::__construct($overviewFactory->createType(self::TYPE),
                            $overviewFactory->createVersion(self::VERSION),
                            $overviewFactory->createIconUsingData(self::ICON),
                            $overviewFactory->useVisualizations()->createText(),
                            $overviewFactory->useOptions()->createOptions(),
                            $overviewFactory->createTitles($overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_GERMAN),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_GERMAN]),
                                                           $overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_ENGLISH),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_ENGLISH])));
    }
    
    
    /**
     * @inheritDoc
     */
    public function data(Options $options): Data
    {
        $visitors = $this->connection->createQueryBuilder()
                        ->select('COUNT(session_id) AS visitors')
                        ->from('whos_online')
                        ->where('FROM_UNIXTIME(time_last_click) BETWEEN DATE_SUB(NOW(), INTERVAL 5 MINUTE) AND NOW()')
                        ->execute()
                        ->fetchAll()[0];
        
        return $this->dataFactory->useTextData()->createTextData($this->dataFactory->useTextData()
                                                                     ->createValue($this->numberFormatter->format((int)(($visitors['visitors']
                                                                                                                         ??
                                                                                                                         0)))));
    }
}

